from datetime import datetime
import json
import logging
import requests
import os
import rel
import time
import websocket

ZOOM_WEBSOCKET_HOST = "zoom-webhook.slideslive.dev"

ZOOM_WEBSOCKET_WORKER_LOG_URL = "https://" + ZOOM_WEBSOCKET_HOST + "/webhooks/-/log"
ZOOM_WEBSOCKET_WORKER_WEBHOOK_LIST_URL = "https://" + ZOOM_WEBSOCKET_HOST + "/webhooks/-/list"
ZOOM_WEBSOCKET_WORKER_MEETING_LIST_URL = "https://" + ZOOM_WEBSOCKET_HOST + "/meetings/-/list"
ZOOM_WEBSOCKET_WORKER_OPEN_URL = "https://" + ZOOM_WEBSOCKET_HOST + "/meetings/{meeting_id}"
ZOOM_WEBSOCKET_WORKER_WEBSOCKET_URL = "wss://" + ZOOM_WEBSOCKET_HOST + "/meetings/{meeting_id}"

VMIX_API_URL = "http://127.0.0.1:8088/API/"


def create_logger():
    datetime_object = datetime.fromtimestamp(time.time())
    formatted_datetime = datetime_object.strftime("%y%m%d_%H%M%S")

    log_dir = os.path.join(os.path.dirname(__file__), "log")
    log_filename = os.path.join(log_dir, f"run_{formatted_datetime}.log")

    os.makedirs(log_dir, exist_ok=True)

    logger = logging.getLogger("zoom_webhook_client")
    logger.setLevel(logging.DEBUG)

    ch = logging.StreamHandler()
    ch.setLevel(logging.DEBUG)

    fh = logging.FileHandler(log_filename)
    fh.setLevel(logging.DEBUG)

    formatter = logging.Formatter("time=%(asctime)s level=%(levelname)s message=%(message)s")
    ch.setFormatter(formatter)
    fh.setFormatter(formatter)

    logger.addHandler(ch)
    logger.addHandler(fh)

    return logger


def call_vmix_api_function(function, value):
    url = f"{VMIX_API_URL}?Function={function}&Value={value}"

    try:
        logger.debug(f"calling vMix API at {url}")
        response = requests.get(url)

        logger.info(f"vMix API response for {url}: status = {response.status_code}")
    except Exception as e:
        logger.error(f"vMix API error for {url}: {e}")


def start_sharing_in_vmix():
    call_vmix_api_function("KeyPress", "C")


def stop_sharing_in_vmix():
    call_vmix_api_function("KeyPress", "X")


def on_open(ws):
    logger.info("websocket connection opened")


def on_message(ws, message):
    global sharing_enabled

    data = json.loads(message)
    zoom_meeting = data["zoomMeeting"]

    logger.debug(f"received websocket data: {data}")

    if sharing_enabled is None:
        logger.info(f"zoom event skipped, new sharing state = {zoom_meeting['sharingEnabled']}")
        sharing_enabled = zoom_meeting["sharingEnabled"]
    elif sharing_enabled == zoom_meeting["sharingEnabled"]:
        logger.info(
            f"zoom event skipped, current sharing state = {sharing_enabled}, zoom sharing state = {zoom_meeting['sharingEnabled']}"
        )
    elif zoom_meeting["sharingEnabled"]:
        logger.info("sharing started")
        sharing_enabled = zoom_meeting["sharingEnabled"]
        start_sharing_in_vmix()
    else:
        logger.info("sharing stopped")
        sharing_enabled = zoom_meeting["sharingEnabled"]
        stop_sharing_in_vmix()


def on_close(ws, close_status_code, close_msg):
    logger.info(f"websocket connection closed ({close_status_code}): {close_msg}")


def on_error(ws, error):
    logger.error(f"websocket connection error: {error}")


def stop_signal_handler():
    global stopped

    logger.error("stop requested")

    stopped = True
    rel.abort()


sharing_enabled = None
stopped = False

logger = create_logger()

logger.info(f"webhook log: {ZOOM_WEBSOCKET_WORKER_LOG_URL}")
logger.info(f"webhook list: {ZOOM_WEBSOCKET_WORKER_WEBHOOK_LIST_URL}")
logger.info(f"meeting list: {ZOOM_WEBSOCKET_WORKER_MEETING_LIST_URL}")

meeting_id = input("Enter Zoom Meeting ID:\n")

logger.info(f"zoom meeting id: {meeting_id}")
logger.info(f"zoom meeting info: {ZOOM_WEBSOCKET_WORKER_OPEN_URL.format(meeting_id=meeting_id)}")

while not stopped:
    try:
        logger.info("starting websocket client")

        ws = websocket.WebSocketApp(
            ZOOM_WEBSOCKET_WORKER_WEBSOCKET_URL.format(meeting_id=meeting_id),
            on_open=on_open,
            on_message=on_message,
            on_close=on_close,
            on_error=on_error,
        )

        ws.run_forever(dispatcher=rel, reconnect=1)

        rel.signal(2, stop_signal_handler)
        rel.signal(15, stop_signal_handler)
        rel.dispatch()

        logger.info("websocket client stopped")
    except Exception as e:
        logger.error(f"EXCEPTION: {e}")

    if not stopped:
        time.sleep(1)
